# -*- coding: UTF-8 -*-
# Copyright 2015-2022 - RoboDK Inc. - https://robodk.com/
# Copyright 2022 - Dmitry Lavygin - vdm.inbox@gmail.com
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# http://www.apache.org/licenses/LICENSE-2.0
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ---------------------------------------------------------------------------------

import sys
import socket

def push_message(message):
    print(message)
    sys.stdout.flush()

def status_message(message):
    push_message('SMS:' + message)

def debug_message(message):
    #push_message('DEBUG: ' + message)
    return

class HanwhaConnection:
    DEFAULT_TIMEOUT = 5
    DEFAULT_IP = '192.168.1.1'
    DEFAULT_PORT = 7000

    def __init__(self):
        self.socket = None
        self.error = ''
    
    def __del__(self):
        self.disconnect()

    def connect(self, ip, port):
        self.disconnect()
        self.error = ''
        self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.socket.settimeout(self.DEFAULT_TIMEOUT)
        if not ip:
            ip = self.DEFAULT_IP
        if not port:
            port = self.DEFAULT_PORT
        try:
            self.socket.connect((ip, port))
        except Exception as e:
            self.disconnect()
            self.error = str(e)
            return False
        return True
    
    def disconnect(self):
        if self.socket:
            try:
                self.socket.close()
            except:
                pass
        self.socket = None

    def is_connected(self):
        return True if self.socket else False

    def short_error(self):
        if len(self.error) > 30: 
            return self.error[:30] + '...'
        else:
            return self.error

    def write_line(self, line):
        if not self.is_connected():
            self.error = 'not connected'
            return False
        debug_message('<write> ' + line)
        data = bytes(line + '\n', 'utf-8')
        result = 0
        try:
            result = self.socket.send(data)
        except Exception as e:
            self.disconnect()
            self.error = str(e)
            return False
        if result < 1:
            self.disconnect()
            self.error = 'unable to send data'
            return False
        return True

    def read_line(self):
        if not self.is_connected():
            self.error = 'not connected'
            return None
        data = b''
        try:
            while True:
                character = self.socket.recv(1)
                if character == b'\n':
                    break
                data += character
        except Exception as e:
            self.disconnect()
            self.error = str(e)
            return None
        result = data.decode('utf-8')
        debug_message('<read> ' + result)
        return result

# Global Variables
g_terminate = False
g_connection = HanwhaConnection()


def execute(line):
    global g_terminate
    global g_connection

    push_message('<stdin> ' + line)

    if len(line) < 1:
        return
    elif line.startswith('CONNECT'):
        arguments = list(filter(None, line.split(' ')))
        ip = None
        port = None
        if len(arguments) > 2:
            try:
                port = int(arguments[2])
            except:
                port = None
        if len(arguments) > 1:
            ip = arguments[1]
        if g_connection.connect(ip, port):
            g_connection.write_line('VERSION')
            status_message('Working...')
            answer = g_connection.read_line()
            if answer and answer.startswith('VERSION'):
                status_message('Ready')
            else:
                g_connection.disconnect()
                status_message('Invalid connector')
        else:
            status_message(f'Unable to connect ({g_connection.short_error()})')
        pass
    elif line.startswith('DISCONNECT'):
        g_connection.disconnect()
        status_message('Disconnected')
    elif line.startswith('STOP') or line.startswith('QUIT'):
        g_connection.disconnect()
        status_message('Stopped')
        g_terminate = True
    elif line.startswith('CJNT'):
        status_message('Working...')
        g_connection.write_line(line)
        answer = g_connection.read_line()
        if answer:
            push_message(answer)
            status_message('Ready')
        else:
            status_message(f'Transmission error ({g_connection.short_error()})')
    elif line.startswith('MOVJ'):
        status_message('Working...')
        g_connection.write_line(line)
        answer = g_connection.read_line()
        success = False
        delay = 0
        if answer:
            while True:
                if not g_connection.write_line('RSTATE CJNT'):
                    break
                # First line: JNTS <...> or JNTS_MOVING <...>
                answer = g_connection.read_line()
                if not answer:
                    break
                # Second line: RSTATE <...>
                if not g_connection.read_line():
                    break
                if answer.startswith('JNTS_MOVING'):
                    push_message(answer)
                    delay = 0
                if answer.startswith('JNTS '):
                    push_message(answer.replace('JNTS', 'JNTS_MOVING', 1))
                    delay = delay + 1
                    if delay > 10:
                        success = True
                        break
        if success:
            status_message('Ready')
        else:
            status_message(f'Transmission error ({g_connection.short_error()})')



def main():
    global g_terminate

    for line in sys.stdin:
        execute(line.rstrip(" \r\n"))
        if g_terminate:
            return

if __name__== "__main__":
    main()